#region References

using System;
using System.Collections;
using System.Data;
using System.Data.SqlClient;
using System.Diagnostics;
using System.Reflection;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.ExceptionManagement;

#endregion

namespace gov.va.med.vbecs.Common
{
    #region Header

    ///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
    ///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
    ///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
    ///<Developers>
    ///	<Developer>Bill Faruki</Developer>
    ///</Developers>
    ///<SiteName>Hines OIFO</SiteName>
    ///	///<CreationDate>7/22/2002</CreationDate>
    ///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
    /// <summary>
    /// Data access layer class representing a SQL Server stored procedure.
    /// Implements functionality allowing excute arbitrary stored 
    /// procedure with chosen input/output parameters.
    /// </summary>

    #endregion


    public class StoredProcedure
    {
        #region Variables

        /// <summary>
        /// Connection Type
        /// </summary>
        private Common.ConnectionType _connectionType = Common.ConnectionType.VBECS;
        private string _overrideConnectionString = string.Empty;    //should only be used by GetVBECSSettingsTable
        private const string PARAM_NAME_PREFIX = "@";
        private const string RETURN_VALUE_PARAMETER_NAME = "ReturnValue";
        private const string LAST_UPDATE_USER_COLUMN_NAME = "LastUpdateUser";
        private const string LAST_UPDATE_DATE_COLUMN_NAME = "LastUpdateDate";
        private static readonly string LAST_UPDATE_USER_PARAMETER_NAME = String.Concat(PARAM_NAME_PREFIX, LAST_UPDATE_USER_COLUMN_NAME);
        private static readonly string LAST_UPDATE_DATE_PARAMETER_NAME = String.Concat(PARAM_NAME_PREFIX, LAST_UPDATE_DATE_COLUMN_NAME);

        private const int ROW_VERSION_VIOLATION_SQL_ERROR_NUMBER = 50001;
        private const int PRIMARY_KEY_VIOLATION_SQL_ERROR_NUMBER = 2627;
        private const int UNIQUE_INDEX_VIOLATION_SQL_ERROR_NUMBER = 2601;
        private const int QUERY_TIMEOUT_SQL_ERROR_NUMBER = -2;          //CR2113 - trapped a timeout several times and it was always this number, so hopefully its the only one

        private const string SQL_CONNECTION_LOST = "20";
        private const string SQL_NETWORK_ERROR = "40";      //CR3460
        private const string SQL_AG_UNAVAILABLE = "14";     //CR3460
        #endregion

        #region Constructors

        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>9/30/2002</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="444"> 
        ///		<ExpectedInput>None</ExpectedInput>
        ///		<ExpectedOutput>StoredProcedure object.</ExpectedOutput>
        ///	</Case>
        ///<Case type="1" testid ="4133"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        ///  Default constructor that sets up a new command object and
        ///  connects it with the default VBECS database
        /// </summary>
        public StoredProcedure() : this(Common.ConnectionType.VBECS) { }

        ///<Developers>
        ///	<Developer>Martin Tsotchev</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/14/2004</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="3646"> 
        ///		<ExpectedInput>Connection Type</ExpectedInput>
        ///		<ExpectedOutput>StoredProcedure Object</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="3647"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="connectionType">Connection type.</param>
        public StoredProcedure(Common.ConnectionType connectionType)
        {
            _connectionType = connectionType;
        }



        /// <summary>
        /// Allows override of the connection string. NOT FOR VBECS CALLS!
        /// </summary>
        /// <param name="connectionType"></param>
        /// <param name="dbConnString"></param>
        public StoredProcedure(Common.ConnectionType connectionType, string dbConnString)
        {
            _connectionType = connectionType;
            _overrideConnectionString = dbConnString;
        }

        #endregion

        #region Abstract Classes / Interfaces

        /// <summary>
        /// ITransactionalWorkUnit
        /// </summary>
        private interface ITransactionalWorkUnit
        {
            int Execute(SqlConnection openConnection, SqlTransaction openTransaction);
        }

        /// <summary>
        /// TransactionalWorkUnitBase
        /// </summary>
        private abstract class TransactionalWorkUnitBase : ITransactionalWorkUnit
        {
            protected readonly SqlCommand _command;
            protected readonly DateTime _transactionDateTime;

            protected DateTime TransactionDateTime
            {
                get
                {
                    return _transactionDateTime;
                }
            }

            protected TransactionalWorkUnitBase()
            {
                _command = new SqlCommand();
                _command.CommandType = CommandType.StoredProcedure;
                _command.CommandTimeout = VbecsConfig.Current.DbQueryTimeoutSeconds;        //no roperty - we wont "work" for non-VBECS

                _transactionDateTime = DateTime.UtcNow;
            }

            public int Execute(SqlConnection openConnection, SqlTransaction openTransaction)
            {
                _command.Connection = openConnection;
                _command.Transaction = openTransaction;

                return ExecuteAction();
            }

            protected abstract int ExecuteAction();

            protected int RunCommandGetReturnValue()
            {
                try
                {
                    _command.ExecuteNonQuery();
                    return (int)_command.Parameters[RETURN_VALUE_PARAMETER_NAME].Value;
                }
                catch (SqlException ex)
                {
                    throw WrapSqlExceptionIntoAppSpecificException(ex);
                }
                catch (Exception ex)
                {
                    throw new DataAccessLayerException(Common.StrRes.SysErrMsg.Common.SystemSaveError(
                        Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name), ex.Message, ex).ResString, ex);
                }
            }
        }

        /// <summary>
        /// Converts <see cref="SqlException"/> caught during transactional DB update call and wraps it 
        /// into application specific exception.
        /// </summary>
        /// <param name="sourceXcp"><see cref="SqlException"/> caught during DB update call. </param>
        /// <returns>Application specific data access exception.</returns>
        private static BaseApplicationException WrapSqlExceptionIntoAppSpecificException(SqlException sourceXcp)
        {
            if (sourceXcp == null)
                throw (new ArgumentNullException("sourceXcp"));

            if (sourceXcp.Errors.Count > 0)
            {
                if (sourceXcp.Errors[0].Number == ROW_VERSION_VIOLATION_SQL_ERROR_NUMBER)
                    throw new RowVersionException(Common.StrRes.SysErrMsg.Common.DataNotSaved("Data ").ResString);

                if (sourceXcp.Errors[0].Number == PRIMARY_KEY_VIOLATION_SQL_ERROR_NUMBER || sourceXcp.Errors[0].Number == UNIQUE_INDEX_VIOLATION_SQL_ERROR_NUMBER)
                    throw (new UniqueDatabaseConstraintViolatedException(sourceXcp.Errors[0].Message, sourceXcp));

                if (IsSQLQueryTimeout(sourceXcp))   //CR2113
                    throw (new SqlTimeoutException());
            }

            throw new DataAccessLayerException(Common.StrRes.SysErrMsg.Common.DataSaveError(
                Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name), sourceXcp.Message, sourceXcp).ResString, sourceXcp);
        }

        /// <summary>
        /// SingleStoredProcWorkUnitBase
        /// </summary>
        private abstract class SingleStoredProcWorkUnitBase : TransactionalWorkUnitBase
        {
            protected SingleStoredProcWorkUnitBase(string storedProcName)
                : base()
            {
                _command.CommandText = storedProcName;
            }
        }

        /// <summary>
        /// SingleCallWorkUnitBase
        /// </summary>
        private abstract class SingleCallWorkUnitBase : SingleStoredProcWorkUnitBase
        {
            protected SingleCallWorkUnitBase(string storedProcName)
                : base(storedProcName) { }

            protected override int ExecuteAction()
            {
                return RunCommandGetReturnValue();
            }
        }

        /// <summary>
        /// SingleCallParamBasedWorkUnit
        /// </summary>
        private class SingleCallParamBasedWorkUnit : SingleCallWorkUnitBase
        {
            public SingleCallParamBasedWorkUnit(string storedProcName, SqlParameter[] parameters)
                : base(storedProcName)
            {
                for (int i = 0; i < parameters.Length; i++)
                {
                    if (!parameters[i].ParameterName.Equals(LAST_UPDATE_DATE_PARAMETER_NAME))
                    {
                        _command.Parameters.Add(parameters[i]);
                    }
                }

                _command.Parameters.Add(MakeLastUpdateDateSqlParam(TransactionDateTime));

                _command.Parameters.Add(MakeReturnValueSqlParam());
            }
        }

        /// <summary>
        /// SingleCallDataRowBasedWorkUnit
        /// </summary>
        private class SingleCallDataRowBasedWorkUnit : SingleCallWorkUnitBase
        {
            public SingleCallDataRowBasedWorkUnit(string storedProcName, DataRow dataRow)
                : base(storedProcName)
            {
                BuildStandardParamSet(_command, dataRow, TransactionDateTime);
            }
        }

        /// <summary>
        /// DataTableBasedWorkUnit
        /// </summary>
        private class DataTableBasedWorkUnit : SingleStoredProcWorkUnitBase
        {
            private readonly DataTable _dataTable;

            public DataTableBasedWorkUnit(string storedProcName, DataTable dataTable)
                : base(storedProcName)
            {
                _dataTable = dataTable;
            }

            protected override int ExecuteAction()
            {
                int _result = 0;

                for (int i = 0; i < _dataTable.Rows.Count; i++)
                {
                    _command.Parameters.Clear();

                    BuildStandardParamSet(_command, _dataTable.Rows[i], TransactionDateTime);

                    _result = RunCommandGetReturnValue();
                }

                return _result;
            }
        }

        /// <summary>
        /// SimpleDataTableBasedWorkUnit: similar to DataTableBasedWorkUnit
        /// but does not append LastUpdateUser information to parameters
        /// </summary>
        private class SimpleDataTableBasedWorkUnit : SingleStoredProcWorkUnitBase
        {
            private readonly DataTable _dataTable;

            public SimpleDataTableBasedWorkUnit(string storedProcName, DataTable dataTable)
                : base(storedProcName)
            {
                _dataTable = dataTable;
            }

            protected override int ExecuteAction()
            {
                int _result = 0;

                for (int i = 0; i < _dataTable.Rows.Count; i++)
                {
                    _command.Parameters.Clear();

                    BuildSimpleParamSet(_command, _dataTable.Rows[i], TransactionDateTime);

                    _result = RunCommandGetReturnValue();
                }

                return _result;
            }
        }

        /// <summary>
        /// EnumeratorsBasedWorkUnit
        /// </summary>
        private class EnumeratorsBasedWorkUnit : TransactionalWorkUnitBase
        {
            IEnumerator _storedProcNamesEnumerator, _dataTablesEnumerator;

            public EnumeratorsBasedWorkUnit(IEnumerable storedProcNames, IEnumerable dataTables)
            {
                _storedProcNamesEnumerator = storedProcNames.GetEnumerator();
                _dataTablesEnumerator = dataTables.GetEnumerator();
            }

            protected override int ExecuteAction()
            {
                int _result = 0;

                while (_storedProcNamesEnumerator.MoveNext() && _dataTablesEnumerator.MoveNext())
                {
                    string _tempStoredProcName = (string)_storedProcNamesEnumerator.Current;

                    DataTable _tempDataTable = (DataTable)_dataTablesEnumerator.Current;

                    if (_tempDataTable == null)
                        continue;

                    _command.CommandText = _tempStoredProcName;

                    for (int i = 0; i < _tempDataTable.Rows.Count; i++)
                    {
                        _command.Parameters.Clear();

                        BuildStandardParamSet(_command, _tempDataTable.Rows[i], TransactionDateTime);

                        _result = RunCommandGetReturnValue();
                    }
                }

                return _result;
            }
        }

        /// <summary>
        /// SimpleEnumeratorsBasedWorkUnit
        /// </summary>
        private class SimpleEnumeratorsBasedWorkUnit : TransactionalWorkUnitBase
        {
            IEnumerator _storedProcNamesEnumerator, _dataTablesEnumerator;

            public SimpleEnumeratorsBasedWorkUnit(IEnumerable storedProcNames, IEnumerable dataTables)
            {
                _storedProcNamesEnumerator = storedProcNames.GetEnumerator();
                _dataTablesEnumerator = dataTables.GetEnumerator();
            }

            protected override int ExecuteAction()
            {
                int _result = 0;

                while (_storedProcNamesEnumerator.MoveNext() && _dataTablesEnumerator.MoveNext())
                {
                    string _tempStoredProcName = (string)_storedProcNamesEnumerator.Current;
                    //
                    DataTable _tempDataTable = (DataTable)_dataTablesEnumerator.Current;
                    //
                    if (_tempDataTable == null)
                    {
                        continue;
                    }
                    //
                    _command.CommandText = _tempStoredProcName;
                    //
                    if (_tempDataTable.Rows.Count == 0)
                    {
                        BuildSimpleParamSet(_command);
                        //
                        _result = RunCommandGetReturnValue();
                    }
                    else
                    {
                        for (int i = 0; i < _tempDataTable.Rows.Count; i++)
                        {
                            _command.Parameters.Clear();
                            //
                            BuildSimpleParamSet(_command, _tempDataTable.Rows[i], TransactionDateTime);
                            //
                            _result = RunCommandGetReturnValue();
                        }
                    }
                }
                //
                return _result;
            }
        }

        /// <summary>
        /// SimpleWorkUnit
        /// CR 2519
        /// </summary>
        private class SimpleWorkUnit : TransactionalWorkUnitBase
        {
            private string _storedProcedureName;

            public SimpleWorkUnit(string storedProcedureName)
            {
                _storedProcedureName = storedProcedureName;
            }

            protected override int ExecuteAction()
            {
                _command.CommandText = _storedProcedureName;
                //
                BuildSimpleParamSet(_command);
                //
                return RunCommandGetReturnValue();
            }
        }

        #endregion

        #region Public Methods

        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>9/30/2002</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="4134"> 
        ///		<ExpectedInput>Sproc Name, Valid Parameters</ExpectedInput>
        ///		<ExpectedOutput>DataTable</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="1346"> 
        ///		<ExpectedInput>Null parameters</ExpectedInput>
        ///		<ExpectedOutput>ArgumentException</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>		
        /// <summary>
        /// Static method retrieving data from a stored procedure with a given name and parameters. 
        /// </summary>
        /// <param name="storedProcName">Name of stored procedure to execute and retrieve data from.</param>
        /// <param name="parameters">Stored procedure parameters.</param>
        /// <returns>Writable <see cref="DataTable"/> populated with data from DB.</returns>
        public static DataTable GetDataTable(string storedProcName, params SqlParameter[] parameters)
        {
            if (storedProcName == null)
                throw (new ArgumentNullException("storedProcName"));

            if (parameters == null)
                throw (new ArgumentNullException("parameters"));

            return new Common.StoredProcedure().RunGetDT(storedProcName, parameters);
        }

        ///<Developers>
        ///	<Developer>Lohse</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>5/06/2013</CreationDate>
        /// <summary>
        /// Will be used for retrieving site configuration values from the VBECS_Support database. 
        /// </summary>
        /// <param name="supportDbConn">Connection string to the VBECS_Support database </param>
        /// <param name="parameters">Stored procedure parameters.</param>
        /// <returns>DataTable</returns>
        public static DataTable GetVBECSSettingsTable(string supportDbConn, params SqlParameter[] parameters)
        {
            if (parameters == null)
                throw (new ArgumentNullException("parameters"));

            return new Common.StoredProcedure(Common.ConnectionType.SettingsReader, supportDbConn).RunGetDT("spGetVbecsSiteList", parameters);
        }

        ///<Developers>
        ///	<Developer>Greg Lohse</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>8/30/2004</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1347"> 
        ///		<ExpectedInput>Sproc Name, Valid Parameters</ExpectedInput>
        ///		<ExpectedOutput>DataRow</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="1348"> 
        ///		<ExpectedInput>Sproc Name, Valid Parameters</ExpectedInput>
        ///		<ExpectedOutput>DataAccessLayerException</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>	
        /// <summary>
        /// Retrieves single <see cref="DataRow"/> from a stored procedure with a given name and parameters. 
        /// Will throw an exception if a number rows returned by stored procedure is not equal to 1. 
        /// </summary>
        /// <param name="storedProcName">Name of stored procedure to execute and retrieve data from.</param>
        /// <param name="parameters">Stored procedure parameters.</param>
        /// <returns>Writable <see cref="DataRow"/> populated with data from DB.</returns>
        public static DataRow GetSingleDataRow(string storedProcName, params SqlParameter[] parameters)
        {
            DataTable _dt = GetDataTable(storedProcName, parameters); // params are verified by callee

            if (_dt.Rows.Count != 1)
                throw (new DataAccessLayerException(StrRes.SysErrMsg.Common.InsufficientNumberOfDataRowsFoundWhileExpectingSingleRow(storedProcName, _dt.Rows.Count).ResString));

            return _dt.Rows[0];
        }

        ///<Developers>
        ///	<Developer>Department of Veterans Affairs</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>3/18/2011</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="9194"> 
        ///		<ExpectedInput>Valid</ExpectedInput>
        ///		<ExpectedOutput></ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="9195"> 
        ///		<ExpectedInput>Invalid</ExpectedInput>
        ///		<ExpectedOutput></ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Retrieves single <see cref="DataRow"/> from a stored procedure with a given name and parameters. 
        /// Will throw an exception if a number tables and/or rows returned by stored procedure is greater than 1.  
        /// CR 2961
        /// </summary>
        /// <param name="storedProcName">Name of stored procedure to execute and retrieve data from.</param>
        /// <returns>Writable <see cref="DataRow"/> populated with data from DB.</returns>
        public static DataRow GetSingleDataRow(string storedProcName)
        {
            if (storedProcName == null)
            {
                throw (new ArgumentNullException("storedProcName"));
            }
            //
            return new Common.StoredProcedure().RunGetDataRow(storedProcName);
        }

        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>9/30/2002</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="441"> 
        ///		<ExpectedInput>Valid sproc name, parameters.</ExpectedInput>
        ///		<ExpectedOutput>Dataset</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="2098"> 
        ///		<ExpectedInput>Invalid parameters</ExpectedInput>
        ///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// A static method that creates an instance of the StoredProcedure object
        /// and calls the RunGetDS function, passing in the name of the stored
        /// procedure to be executed and, optionally, a list of parameters
        /// </summary>
        /// <param name="storedProcName">Stored procedure name</param>
        /// <param name="parameters">Optional stored procedure parameters</param>
        /// <returns>DataSet containing data returned by stored procedure</returns>
        public static DataSet GetData(string storedProcName, params SqlParameter[] parameters)
        {
            if (storedProcName == null)
                throw (new ArgumentNullException("storedProcName"));

            if (parameters == null)
                throw (new ArgumentNullException("parameters"));

            return new Common.StoredProcedure().RunGetDS(storedProcName, parameters);
        }

        ///<Developers>
        ///	<Developer>Greg Lohse</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>8/30/2004</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1349"> 
        ///		<ExpectedInput>Sproc Name, Connection Type, Valid Parameters</ExpectedInput>
        ///		<ExpectedOutput>DataSet</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="1350"> 
        ///		<ExpectedInput>Null Sproc Name, Parameters</ExpectedInput>
        ///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>	
        /// <summary>
        /// A static method that creates an instance of the StoredProcedure object
        /// with a specified connection type
        /// and calls the RunGetDS function, passing in the name of the stored
        /// procedure to be executed and, optionally, a list of parameters
        /// </summary>
        /// <param name="storedProcName">Stored procedure name</param>
        /// <param name="connType">The type of connection to be established</param>
        /// <param name="parameters">Optional stored procedure parameters</param>
        /// <returns>DataSet containing data returned by stored procedure</returns>
        public static DataSet GetData(string storedProcName, ConnectionType connType, params SqlParameter[] parameters)
        {
            if (storedProcName == null)
                throw (new ArgumentNullException("storedProcName"));

            if (parameters == null)
                throw (new ArgumentNullException("parameters"));

            return new Common.StoredProcedure(connType).RunGetDS(storedProcName, parameters);
        }

        ///<Developers>
        ///	<Developer>D. Askew</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>3/8/2006</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="8404"> 
        ///		<ExpectedInput>Stored Procedure Name</ExpectedInput>
        ///		<ExpectedOutput>DataSet</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="8405"> 
        ///		<ExpectedInput>Null</ExpectedInput>
        ///		<ExpectedOutput>Argument Null Exception</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Creates an instance of the StoredProcedure object and calls 
        /// the RunGetDS function, passing in the name of the stored 
        /// procedure to be executed
        /// </summary>
        /// <param name="storedProcName">Stored procedure name</param>
        /// <returns>DataSet containing data returned by stored procedure</returns>
        public static DataSet GetData(string storedProcName)
        {
            if (storedProcName == null)
                throw (new ArgumentNullException("storedProcName"));

            return new Common.StoredProcedure().RunGetDS(storedProcName);
        }

        ///<Developers>
        ///	<Developer>David Askew</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>08/02/2005</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="9186"> 
        ///		<ExpectedInput>Valid</ExpectedInput>
        ///		<ExpectedOutput></ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="9187"> 
        ///		<ExpectedInput>Invalid</ExpectedInput>
        ///		<ExpectedOutput></ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Runs stored procedure and retrieves boolean return value.
        /// Non-transactional; for query-type calls.
        /// CR 2960 (made public)
        /// </summary>
        /// <param name="storedProcName">stored procedure name</param>
        /// <param name="parameters">stored procedure parameters</param>
        /// <returns>return value</returns>
        public bool GetBoolReturnValue(string storedProcName, params SqlParameter[] parameters)
        {
            if (storedProcName == null)
                throw (new ArgumentNullException("storedProcName"));

            if (parameters == null)
                throw (new ArgumentNullException("parameters"));

            return Convert.ToBoolean(RunGetReturnValue(storedProcName, parameters));
        }

        ///<Developers>
        ///	<Developer>David Askew</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>08/02/2005</CreationDate>
        /// <summary>
        /// Runs stored procedure and retrieves boolean return value.
        /// Non-transactional; for query-type calls.
        /// </summary>
        /// <param name="storedProcName">stored procedure name</param>
        /// <param name="dataTable">dataTable consisting of stored procedure parameters</param>
        /// <returns>return value</returns>
        internal bool GetBoolReturnValue(string storedProcName, DataTable dataTable)
        {
            if (storedProcName == null)
                throw (new ArgumentNullException("storedProcName"));

            if (dataTable == null)
                throw (new ArgumentNullException("dataTable"));

            return Convert.ToBoolean(RunGetReturnValue(storedProcName, dataTable));
        }

        ///<Developers>
        ///	<Developer>David Askew</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>08/02/2005</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="9168"> 
        ///		<ExpectedInput>Valid</ExpectedInput>
        ///		<ExpectedOutput></ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="9169"> 
        ///		<ExpectedInput>Invalid</ExpectedInput>
        ///		<ExpectedOutput></ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Runs stored procedure and retrieves integer return value.
        /// Non-transactional; for query-type calls.
        /// CR 2962 (made public)
        /// </summary>
        /// <param name="storedProcName">stored procedure name</param>
        /// <param name="parameters">stored procedure parameters</param>
        /// <returns>return value</returns>
        public int GetIntReturnValue(string storedProcName, params SqlParameter[] parameters)
        {
            if (storedProcName == null)
                throw (new ArgumentNullException("storedProcName"));

            if (parameters == null)
                throw (new ArgumentNullException("parameters"));

            return Convert.ToInt32(RunGetReturnValue(storedProcName, parameters));
        }

        ///<Developers>
        ///	<Developer>David Askew</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>08/02/2005</CreationDate>
        /// <summary>
        /// Runs stored procedure and retrieves integer return value.
        /// Non-transactional; for query-type calls.
        /// </summary>
        /// <param name="storedProcName">stored procedure name</param>
        /// <param name="dataTable">dataTable consisting of stored procedure parameters</param>
        /// <returns>return value</returns>
        internal int GetIntReturnValue(string storedProcName, DataTable dataTable)
        {
            if (storedProcName == null)
                throw (new ArgumentNullException("storedProcName"));

            if (dataTable == null)
                throw (new ArgumentNullException("dataTable"));

            return Convert.ToInt32(RunGetReturnValue(storedProcName, dataTable));
        }

        ///<Developers>
        ///	<Developer>Greg Lohse</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>8/30/2004</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1351"> 
        ///		<ExpectedInput>Sproc Names, DataTables</ExpectedInput>
        ///		<ExpectedOutput>int</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="1354"> 
        ///		<ExpectedInput>Null Sproc Names, DataTables</ExpectedInput>
        ///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>	
        ///<summary>
        /// Executes all specified stored procedures to update (save to database) 
        /// data in all passed DataTables
        ///</summary>
        ///<param name="storedProcNames">ArrayList of stored procedure names</param>
        ///<param name="dataTables">ArrayList of DataTables with data to save</param>
        ///<returns>Last stored procedure return value</returns>
        public int TransactionalGetValue(IEnumerable storedProcNames, IEnumerable dataTables)
        {
            if (storedProcNames == null)
                throw (new ArgumentNullException("storedProcNames"));

            if (dataTables == null)
                throw (new ArgumentNullException("dataTables"));

            return ExecuteTransactionalWorkUnit(new EnumeratorsBasedWorkUnit(storedProcNames, dataTables));
        }

        ///<Developers>
        ///	<Developer>Greg Lohse</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>8/30/2004</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1355"> 
        ///		<ExpectedInput>Sproc Name, DataTable</ExpectedInput>
        ///		<ExpectedOutput>DataRow</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="1357"> 
        ///		<ExpectedInput>Null Sproc Name, DataTable</ExpectedInput>
        ///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>	
        ///<summary>
        /// Executes specified stored procedures to update (save to database) 
        /// data in a given DataTable
        ///</summary>
        ///<param name="storedProcName">Stored procedure name</param>
        ///<param name="dataTable">DataTable with data to save</param>
        ///<returns>Stored procedure return value</returns>
        public int TransactionalGetValue(string storedProcName, DataTable dataTable)
        {
            if (storedProcName == null)
                throw (new ArgumentNullException("storedProcName"));

            if (dataTable == null)
                throw (new ArgumentNullException("dataTable"));

            return ExecuteTransactionalWorkUnit(new DataTableBasedWorkUnit(storedProcName, dataTable));
        }

        ///<Developers>
        ///	<Developer>Greg Lohse</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>8/30/2004</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1358"> 
        ///		<ExpectedInput>Sproc Name, Valid DataRow</ExpectedInput>
        ///		<ExpectedOutput>int</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="1359"> 
        ///		<ExpectedInput>Null Sproc Name, DataRow</ExpectedInput>
        ///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>	
        ///<summary>
        /// Executes specified stored procedures to update (save to database) 
        /// data in a given DataRow
        ///</summary>
        ///<param name="storedProcName">Stored procedure name</param>
        ///<param name="dataRow">DataRow with data to save</param>
        ///<returns>Stored procedure return value</returns>
        public int TransactionalGetValue(string storedProcName, DataRow dataRow)
        {
            if (storedProcName == null)
                throw (new ArgumentNullException("storedProcName"));

            if (dataRow == null)
                throw (new ArgumentNullException("dataRow"));

            return ExecuteTransactionalWorkUnit(new SingleCallDataRowBasedWorkUnit(storedProcName, dataRow));
        }

        ///<Developers>
        ///	<Developer>Martin Tsotchev</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>1/12/2004</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="3612"> 
        ///		<ExpectedInput>Valid sprocname, parameters</ExpectedInput>
        ///		<ExpectedOutput>int</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="3613"> 
        ///		<ExpectedInput>Null sprocName, parameters</ExpectedInput>
        ///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        /// <summary>
        /// Executes the specified stored procedure and returns a stored procedure return value.
        /// </summary>
        /// <param name="storedProcName">Stored procedure name</param>
        /// <param name="parameters">Optional stored procedure parameters</param>
        /// <returns>Stored procedure return value</returns>
        public int TransactionalGetValue(string storedProcName, params SqlParameter[] parameters)
        {
            if (storedProcName == null)
                throw (new ArgumentNullException("storedProcName"));

            if (parameters == null)
                throw (new ArgumentNullException("parameters"));

            return ExecuteTransactionalWorkUnit(new SingleCallParamBasedWorkUnit(storedProcName, parameters));
        }

        ///<Developers>
        ///	<Developer>D. Askew</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>02/16/2006</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="8393"> 
        ///		<ExpectedInput>SprocName, DataTable</ExpectedInput>
        ///		<ExpectedOutput>int</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="8394"> 
        ///		<ExpectedInput>Invalid parameters</ExpectedInput>
        ///		<ExpectedOutput>SqlException</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>	
        ///<summary>
        /// Executes specified stored procedures to insert or update
        /// data in a given DataTable; does not append any additional
        /// parameters to dataTable other than LastUpdateDate 
        ///</summary>
        ///<param name="storedProcName">Stored procedure name</param>
        ///<param name="dataTable">DataTable with data to save</param>
        ///<returns>Stored procedure return value</returns>
        public int SimpleTransactionalGetValue(string storedProcName, DataTable dataTable)
        {
            if (storedProcName == null)
                throw (new ArgumentNullException("storedProcName"));

            if (dataTable == null)
                throw (new ArgumentNullException("dataTable"));

            return ExecuteTransactionalWorkUnit(new SimpleDataTableBasedWorkUnit(storedProcName, dataTable));
        }

        ///<Developers>
        ///	<Developer>D. Askew</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>3/8/2006</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="8408"> 
        ///		<ExpectedInput>Valid Inputs</ExpectedInput>
        ///		<ExpectedOutput>0</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="8409"> 
        ///		<ExpectedInput>Invalid Inputs</ExpectedInput>
        ///		<ExpectedOutput>Argument Exception</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        ///<summary>
        /// Executes specified stored procedures to insert or update
        /// data in a given DataTable; does not append any additional
        /// parameters to dataTable other than LastUpdateDate 
        ///</summary>
        /// <param name="storedProcNames">Stored procedure name</param>
        /// <param name="dataTables">Stored procedure datatable parameters</param>
        /// <returns>Stored procedure return value</returns>
        public int SimpleTransactionalGetValue(IEnumerable storedProcNames, IEnumerable dataTables)
        {
            if (storedProcNames == null)
                throw (new ArgumentNullException("storedProcNames"));

            if (dataTables == null)
                throw (new ArgumentNullException("dataTables"));

            return ExecuteTransactionalWorkUnit(new SimpleEnumeratorsBasedWorkUnit(storedProcNames, dataTables));
        }

        ///<Developers>
        ///	<Developer>saic</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>6/17/2008</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="8825"> 
        ///		<ExpectedInput>Valid sproc</ExpectedInput>
        ///		<ExpectedOutput>int</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="8826"> 
        ///		<ExpectedInput>null</ExpectedInput>
        ///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        ///
        ///<summary>
        /// Executes specified parameter-less stored procedure 
        /// to insert or update data in a given DataTable 
        /// First used for CR 2519
        ///</summary>
        /// <returns>Stored procedure return value</returns>
        public int SimpleTransactionalGetValue(string storedProcedureName)
        {
            if (storedProcedureName == null)
            {
                throw (new ArgumentNullException("storedProcedureName"));
            }
            //
            return ExecuteTransactionalWorkUnit(new SimpleWorkUnit(storedProcedureName));
        }

        ///<Developers>
        ///	<Developer>Greg Lohse</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>8/30/2004</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1363"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>SqlParameter</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="1364"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>	
        /// <summary>
        /// Makes and returns standard 'last update user' sproc parameter.
        /// </summary>
        /// <returns></returns>
        public static SqlParameter MakeLastUpdateUserSqlParam()
        {
            SqlParameter _param = new SqlParameter(LAST_UPDATE_USER_PARAMETER_NAME, Common.LogonUser.LogonUserName);
            _param.DbType = DbType.String;

            return _param;
        }

        ///<Developers>
        ///	<Developer>Greg Lohse</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>8/30/2004</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="1365"> 
        ///		<ExpectedInput>Parameter Name, value</ExpectedInput>
        ///		<ExpectedOutput>SqlParameter</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="1369"> 
        ///		<ExpectedInput>NA</ExpectedInput>
        ///		<ExpectedOutput>NA</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>	
        ///<summary>
        ///	Creates and returns stored procedure parameter (as a SqlParameter)
        ///	from a given parameter name and value.
        ///</summary>
        ///<param name="parameterName">Parameter name</param>
        ///<param name="parameterValue">Parameter value</param>
        ///<returns>Newly created SqlParameter</returns>
        public static SqlParameter MakeParameter(string parameterName, object parameterValue)
        {
            SqlParameter _param = new SqlParameter(parameterName, Common.Utility.GetColumnType(parameterValue.GetType()));
            _param.Value = parameterValue;

            return _param;
        }

        #endregion

        #region Private Methods

        /// <summary>
        /// Make a SQL connection based on the type of connection specified
        /// through the _connectionType property
        /// </summary>
        /// <returns></returns>
        private SqlConnection GetNewConnection()
        {
            if (this._connectionType == ConnectionType.SettingsReader)
            {
                return (new SqlConnection(this._overrideConnectionString));
            }
            else
            {
                return new SqlConnection(VbecsConfig.Current.PrimaryDbConnectionString);
            }

        }

        ///<Developers>
        ///	<Developer>John Vrooland</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>9/30/2002</CreationDate>
        /// <summary>
        /// Executes the specified stored procedure with parameters passed in and 
        /// returns a DataSet with retrieved data
        /// </summary>
        /// <param name="storedProcName">Stored procedure name</param>
        /// <param name="parameters">Optional stored procedure parameters</param>
        /// <returns>A data set of records returned by the stored procedure</returns>
        private DataSet RunGetDS(string storedProcName, params SqlParameter[] parameters)
        {
            DataSet _ds = new DataSet();
            _ds.EnforceConstraints = false;
            PopulateDataObject(storedProcName, parameters, _ds);

            if (_ds.Tables.Count == 0)
                throw new DataAccessLayerException(Common.StrRes.SysErrMsg.Common.NoTablesFoundException(
                    Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name)).ResString);

            // Read-only fix - removing read-only indicator from columns
            foreach (DataTable _dt in _ds.Tables)
            {
                _dt.Constraints.Clear();
                MakeDataTableColumnsWritable(_dt);
            }
            return _ds;
        }

        ///<Developers>
        ///	<Developer>D. Askew</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>3/8/2006</CreationDate>
        ///
        /// <summary>
        /// No parameters version
        /// </summary>
        private DataSet RunGetDS(string storedProcName)
        {
            DataSet _ds = new DataSet();
            _ds.EnforceConstraints = false;
            PopulateDataObject(storedProcName, _ds);

            if (_ds.Tables.Count == 0)
                throw new DataAccessLayerException(Common.StrRes.SysErrMsg.Common.NoTablesFoundException(
                    Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name)).ResString);

            // Read-only fix - removing read-only indicator from columns
            foreach (DataTable _dt in _ds.Tables)
            {
                _dt.Constraints.Clear();
                MakeDataTableColumnsWritable(_dt);
            }
            return _ds;
        }

        /// <summary>
        /// Parameterless version; throws no exception if no
        /// rows are returned; returns single data row
        /// CR 2961
        /// </summary>
        private DataRow RunGetDataRow(string storedProcName)
        {
            DataSet ds = new DataSet();
            //
            ds.EnforceConstraints = false;
            PopulateDataObject(storedProcName, ds);
            //
            if (ds.Tables.Count > 0)
            {
                if (ds.Tables.Count > 1)
                {
                    throw new DataAccessLayerException(string.Concat(ds.Tables.Count, " tables returned by ", storedProcName, ".  One table was expected."));
                }
                //
                if (ds.Tables[0].Rows.Count > 1)
                {
                    throw new DataAccessLayerException(string.Concat(ds.Tables[0].Rows.Count, " rows returned by ", storedProcName, ".  One row was expected."));
                }
                //
                // Read-only fix - removing read-only indicator from columns
                MakeDataTableColumnsWritable(ds.Tables[0]);
                //
                return ds.Tables[0].Rows[0];
            }
            //
            return null;
        }

        /// <summary>
        /// MakeDataTableColumnsWritable
        /// </summary>
        private static void MakeDataTableColumnsWritable(DataTable dataTable)
        {
            foreach (DataColumn _col in dataTable.Columns)
                if (_col.ReadOnly)
                    _col.ReadOnly = false;
        }

        /// <summary>
        /// PrepareSqlGetCommand
        /// </summary>
        private SqlCommand PrepareSqlGetCommand(SqlConnection conn, string storedProcName, SqlParameter[] parameters)
        {
            if (conn == null)
                throw (new ArgumentNullException("conn"));

            SqlCommand _cmd = conn.CreateCommand();

            _cmd.CommandText = storedProcName;
            _cmd.CommandType = CommandType.StoredProcedure;
            _cmd.CommandTimeout = this.DBQueryTimeout;

            foreach (SqlParameter _parameter in parameters)
                _cmd.Parameters.Add(_parameter);

            _cmd.Parameters.Add(MakeReturnValueSqlParam());

            return _cmd;
        }

        ///<Developers>
        ///	<Developer>D. Askew</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>3/8/2006</CreationDate>
        ///
        /// <summary>
        /// No parameters version
        /// </summary>
        private SqlCommand PrepareSqlGetCommand(SqlConnection conn, string storedProcName)
        {
            if (conn == null)
                throw (new ArgumentNullException("conn"));

            SqlCommand _cmd = conn.CreateCommand();

            _cmd.CommandText = storedProcName;
            _cmd.CommandType = CommandType.StoredProcedure;
            _cmd.CommandTimeout = this.DBQueryTimeout;

            _cmd.Parameters.Add(MakeReturnValueSqlParam());

            return _cmd;
        }

        /// <summary>
        /// RunGetDT
        /// </summary>
        private DataTable RunGetDT(string storedProcName, params SqlParameter[] parameters)
        {
            DataTable _dt = new DataTable();

            PopulateDataObject(storedProcName, parameters, _dt);

            MakeDataTableColumnsWritable(_dt);
            _dt.Constraints.Clear();
            return _dt;
        }

        // WARNING! FillSchema was removed here (Stas) and replaced with MissingSchemaAction. 
        // Techically, this should work, but practically can lead to bugz. 
        private object PopulateDataObject(string storedProcName, SqlParameter[] parameters, object dataObject)
        {
            try //CR3460
            {
                using (SqlConnection _cn = GetNewConnection())
                {
                    SqlCommand _cmd = PrepareSqlGetCommand(_cn, storedProcName, parameters);

                    using (SqlDataAdapter _dataAdapter = new SqlDataAdapter(_cmd))
                    {
                        _dataAdapter.MissingSchemaAction = MissingSchemaAction.AddWithKey;


                        _cn.Open();

                        if (dataObject is DataTable)
                        {
                            DataTable _dtTable = ((DataTable)dataObject);
                            _dtTable.Constraints.Clear();
                            _dataAdapter.Fill(_dtTable);
                        }
                        else
                        {
                            System.Data.DataSet dsTemp = ((DataSet)dataObject);
                            dsTemp.EnforceConstraints = false;
                            foreach (DataTable _dt in dsTemp.Tables)
                            {
                                _dt.Constraints.Clear();
                            }
                            _dataAdapter.Fill(dsTemp);
                        }
                        return dataObject;
                    }
                }
            }
            catch (System.Data.SqlClient.SqlException expSql)
            {
                string msg = string.Empty;
                bool failover = false;
                for (int i = 0; i < expSql.Errors.Count; i++)
                {
                    msg = expSql.Errors[i].Message.ToString();
                    //The SqlConnection remains open when the severity level is 19 or less. When the severity level is 20 or greater, the server usually closes the SqlConnection. However, the user can reopen the connection and continue. In both cases, a SqlException is generated by the method executing the command.
                    if (IsSQLConnectionError(expSql))  //CR3460
                    {
                        failover = true;
                        break;
                    }

                }
                if (failover)
                {
                    throw new SqlConnectionException(msg);
                }
                else
                {
                    if (IsSQLQueryTimeout(expSql))        //CR2113 - we should really use WrapSqlExceptionIntoAppSpecificException instead
                    {
                        throw new SqlTimeoutException();
                    }
                    throw;
                }
            }
        }

        ///<Developers>
        ///	<Developer>D. Askew</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>3/8/2006</CreationDate>
        ///
        /// <summary>
        /// No parameters version
        /// </summary>
        private object PopulateDataObject(string storedProcName, object dataObject)
        {
            try //CR3460
            {
                using (SqlConnection _cn = GetNewConnection())
                {
                    SqlCommand _cmd = PrepareSqlGetCommand(_cn, storedProcName);

                    using (SqlDataAdapter _dataAdapter = new SqlDataAdapter(_cmd))
                    {
                        _dataAdapter.MissingSchemaAction = MissingSchemaAction.AddWithKey;
                        _cn.Open();

                        if (dataObject is DataTable)
                        {
                            DataTable _dt = ((DataTable)dataObject);
                            _dt.Constraints.Clear();
                            _dataAdapter.Fill(_dt);
                        }
                        else
                        {
                            System.Data.DataSet dsTemp = ((DataSet)dataObject);
                            dsTemp.EnforceConstraints = false;
                            foreach (DataTable _dt in dsTemp.Tables)
                            {
                                _dt.Constraints.Clear();
                            }
                            _dataAdapter.Fill(dsTemp);
                        }
                        return dataObject;
                    }
                }
            }
            catch (System.Data.SqlClient.SqlException expSql)
            {
                string msg = string.Empty;
                bool failover = false;
                for (int i = 0; i < expSql.Errors.Count; i++)
                {
                    msg = expSql.Errors[i].Message.ToString();

                    //The SqlConnection remains open when the severity level is 19 or less. When the severity level is 20 or greater, the server usually closes the SqlConnection. However, the user can reopen the connection and continue. In both cases, a SqlException is generated by the method executing the command.
                    if (IsSQLConnectionError(expSql)) //CR3460
                    {
                        failover = true;
                        break;
                    }

                }
                if (failover)
                {
                    throw new SqlConnectionException(msg);
                }
                else
                {
                    if (IsSQLQueryTimeout(expSql))        //CR2113 - we should really use WrapSqlExceptionIntoAppSpecificException instead
                    {
                        throw new SqlTimeoutException();
                    }
                    throw;
                }

            }
        }

        /// <summary>
        /// ExecuteTransactionalWorkUnit
        /// </summary>
        private int ExecuteTransactionalWorkUnit(ITransactionalWorkUnit workUnit)
        {
            if (workUnit == null)
                throw (new ArgumentNullException("workUnit"));

            SqlTransaction _trn = null;
            SqlConnection _cn = null;
            try //CR3460
            {
                _cn = GetNewConnection();

                _cn.Open();

                _trn = _cn.BeginTransaction();

                int _result = workUnit.Execute(_cn, _trn);

                _trn.Commit();

                return _result;
            }
            catch (DataAccessLayerException exc)
            {
                //CR2077 - trap deadlocks and treat them as RowVersionExceptions'
                //RowVersionException errors will not be caught in this catch block, the are already caught in workUnit class and
                //will bubble up to the BOL -- (see WrapSqlExceptionIntoAppSpecificException() method)
                SqlException sqlErr = (SqlException)exc.InnerException;

                bool deadlock = sqlErr.Number == 1205; //1205 = deadlock

                if (_trn != null) _trn.Rollback();

                //Put logic here to determine if the save is associated with the save of blood unit status
                //if it is associated, try again, if not, raise row version exception
                if (deadlock)
                {
                    throw new RowVersionException(Common.StrRes.SysErrMsg.Common.DataNotSaved("Data ").ResString);
                }
                else
                {
                    if (IsSQLConnectionError(sqlErr))       //CR3460
                    {
                        throw (new SqlConnectionException(sqlErr.Message));
                    }
                    else
                    {
                        if (IsSQLQueryTimeout(sqlErr))        //CR2113 - we should really use WrapSqlExceptionIntoAppSpecificException instead
                        {
                            throw new SqlTimeoutException();
                        }
                        throw;
                    }
                }
            }
            finally //CR3460
            {
                if (_cn != null)
                {
                    _cn.Close();
                }
            }
        }

        //CR2113 - for readability and customizing
        private static bool IsSQLQueryTimeout(SqlException trgException)
        {
            return (trgException.Number == QUERY_TIMEOUT_SQL_ERROR_NUMBER);
        }

        //CR3460 - for readability and customizing
        private static bool IsSQLConnectionError(SqlException trgException)
        {
            if (trgException.Number == PRIMARY_KEY_VIOLATION_SQL_ERROR_NUMBER ||
                trgException.Number == QUERY_TIMEOUT_SQL_ERROR_NUMBER ||
                trgException.Number == ROW_VERSION_VIOLATION_SQL_ERROR_NUMBER ||
                trgException.Number == UNIQUE_INDEX_VIOLATION_SQL_ERROR_NUMBER)
            {
                return (false);             
            }
            else
            {
                string cls = trgException.Class.ToString();
                return (cls == SQL_CONNECTION_LOST || cls == SQL_NETWORK_ERROR || cls == SQL_AG_UNAVAILABLE);
            }
        }

        /// <summary>
        /// BuildStandardParamSet : used for return value queries
        /// </summary>
        private static SqlCommand BuildStandardParamSet(SqlCommand command, DataRow dataRow)
        {
            for (int i = 0; i < dataRow.Table.Columns.Count; i++)
            {
                DataColumn column = dataRow.Table.Columns[i];

                if (dataRow[column] == null || (dataRow[column].GetType() == typeof(System.DBNull)))
                    continue;

                SqlParameter param = new SqlParameter
                (
                    String.Concat(PARAM_NAME_PREFIX, column.ColumnName),
                    Utility.GetColumnType(dataRow[column].GetType())
                );

                param.Value = dataRow[column];

                command.Parameters.Add(param);
            }

            command.Parameters.Add(MakeReturnValueSqlParam());

            return command;
        }

        /// <summary>
        /// BuildStandardParamSet : used for updates, requires lastUpdateDate
        /// </summary>
        private static SqlCommand BuildStandardParamSet(SqlCommand command, DataRow dataRow, DateTime lastUpdateDate)
        {
            for (int i = 0; i < dataRow.Table.Columns.Count; i++)
            {
                DataColumn column = dataRow.Table.Columns[i];

                if (dataRow[column] == null || (dataRow[column].GetType() == typeof(System.DBNull)))
                    continue;

                SqlParameter param = new SqlParameter
                (
                    String.Concat(PARAM_NAME_PREFIX, column.ColumnName),
                    Utility.GetColumnType(dataRow[column].GetType())
                );

                param.Value = dataRow[column];

                // Never use what the caller has passed in as the update user 
                // always get it from the environment.username here centrally
                // did not want to prevent people from retrieving the column because
                // it is needed in some use cases but the values they set in that field should never
                // be different than the environment.username
                // ----------------------------------------------------------------------------------
                // Also, ignore passed in lastUpdateDate because 
                // this is now set for the entire transaction
                if (column.ColumnName != LAST_UPDATE_USER_COLUMN_NAME && column.ColumnName != LAST_UPDATE_DATE_COLUMN_NAME)
                {
                    command.Parameters.Add(param);
                }
            }

            command.Parameters.Add(MakeLastUpdateUserSqlParam());

            command.Parameters.Add(MakeLastUpdateDateSqlParam(lastUpdateDate));

            command.Parameters.Add(MakeReturnValueSqlParam());

            return command;
        }

        /// <summary>
        /// BuildSimpleParamSet: used for updates, requires lastUpdateDate; does not append lastUpdateUser
        /// </summary>
        private static SqlCommand BuildSimpleParamSet(SqlCommand command, DataRow dataRow, DateTime lastUpdateDate)
        {
            for (int i = 0; i < dataRow.Table.Columns.Count; i++)
            {
                DataColumn column = dataRow.Table.Columns[i];
                //
                if (dataRow[column] == null || (dataRow[column].GetType() == typeof(System.DBNull)))
                {
                    continue;
                }
                //
                SqlParameter param = new SqlParameter
                (
                    String.Concat(PARAM_NAME_PREFIX, column.ColumnName),
                    Utility.GetColumnType(dataRow[column].GetType())
                );
                //
                param.Value = dataRow[column];
                //
                // LastUpdateDate is now set for the entire transaction
                if (column.ColumnName != LAST_UPDATE_USER_COLUMN_NAME && column.ColumnName != LAST_UPDATE_DATE_COLUMN_NAME)
                {
                    command.Parameters.Add(param);
                }
            }
            //
            command.Parameters.Add(MakeLastUpdateDateSqlParam(lastUpdateDate));
            //
            command.Parameters.Add(MakeReturnValueSqlParam());
            //
            return command;
        }

        /// <summary>
        /// BuildSimpleParamSet: used for updates; does not expect any existing
        /// parameters
        /// </summary>
        private static SqlCommand BuildSimpleParamSet(SqlCommand command)
        {
            command.Parameters.Add(MakeReturnValueSqlParam());
            //
            return command;
        }

        /// <summary>
        /// Time in Seconds for a query to execute before giving up
        /// </summary>
        private int DBQueryTimeout
        {
            get
            {
                if (this._connectionType == ConnectionType.VBECS)
                {
                    return (VbecsConfig.Current.DbQueryTimeoutSeconds);
                }
                else
                {
                    return (90);
                }
            }
        }

        /// <summary>
        /// Runs stored procedure and retrieves return value as object.
        /// Non-transactional; for query-type calls. 
        /// </summary>
        /// <param name="storedProcName">stored procedure name</param>
        /// <param name="parameterObject">stored procedure parameters or dataTable consisting of stored procedure parameters</param>
        /// <returns>return value</returns>
        private object RunGetReturnValue(string storedProcName, object parameterObject)
        {
            using (SqlConnection _cn = GetNewConnection())
            {
                if (_cn == null)
                    throw (new ArgumentNullException("connection"));

                try
                {
                    SqlCommand _command = _cn.CreateCommand();

                    _command.CommandText = storedProcName;
                    _command.CommandType = CommandType.StoredProcedure;
                    _command.CommandTimeout = this.DBQueryTimeout;

                    _cn.Open();

                    if (parameterObject is DataTable)
                    {
                        DataTable paramaterTable = (DataTable)parameterObject;

                        for (int i = 0; i < paramaterTable.Rows.Count; i++)
                        {
                            _command.Parameters.Clear();

                            BuildStandardParamSet(_command, paramaterTable.Rows[i]);

                            _command.ExecuteNonQuery();
                        }
                    }
                    else
                    {
                        SqlParameter[] parameters = (SqlParameter[])parameterObject;

                        for (int i = 0; i < parameters.Length; i++)
                        {
                            _command.Parameters.Add(parameters[i]);
                        }

                        _command.Parameters.Add(MakeReturnValueSqlParam());

                        _command.ExecuteNonQuery();
                    }
                    //
                    return _command.Parameters[RETURN_VALUE_PARAMETER_NAME].Value;
                }
                catch (Exception ex)
                {
                    throw new DataAccessLayerException(Common.StrRes.SysErrMsg.Common.SystemRetrieveError(
                        Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name), ex.Message, ex).ResString, ex);
                }
            }
        }

        /// <summary>
        /// MakeReturnValueSqlParam
        /// </summary>
        private static SqlParameter MakeReturnValueSqlParam()
        {
            return new SqlParameter(RETURN_VALUE_PARAMETER_NAME, SqlDbType.Int, 4, ParameterDirection.ReturnValue, false, 0, 0, string.Empty, DataRowVersion.Default, null);
        }

        /// <summary>
        /// Makes and returns standard 'last update date' sproc parameter.
        /// </summary>
        /// <returns></returns>
        private static SqlParameter MakeLastUpdateDateSqlParam(DateTime lastUpdateDate)
        {
            SqlParameter param = new SqlParameter(LAST_UPDATE_DATE_PARAMETER_NAME, lastUpdateDate);
            param.DbType = DbType.DateTime;
            //
            return param;
        }

        #endregion
    }
}